/**
 * \file pappsomspp/processing/specglob/peptidemodel.cpp
 * \date 05/10/2025
 * \author Olivier Langella
 * \brief SpecPeptidOMS peptide model
 * \todo https://www.psidev.info/proforma
 *
 */


/*
 * SpecGlobTool, Spectra to peptide alignment tool
 * Copyright (C) 2025  Olivier Langella
 * <olivier.langella@universite-paris-saclay.fr>
 *
 * This program is free software: you can redistribute ipetide to spectrum
 * alignmentt and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "peptidemodel.h"

namespace pappso
{
namespace specpeptidoms
{

QString
AminoAcidModel::toProForma() const
{
  if(m_massDifference != 0)
    {
      return QString("%1[%2]")
        .arg(m_aminoAcid.toProForma())
        .arg(PeptideModel::toProFormaMass(m_massDifference));
    }
  else
    {
      return m_aminoAcid.toProForma();
    }
}

QString
AminoAcidModel::toInterpretation() const
{
  if(m_massDifference != 0)
    {
      if(m_skipped)
        {
          return QString("[%1][%2]")
            .arg(m_aminoAcid.toString())
            .arg(QString::number(m_massDifference));
        }
      else
        {
          return QString("%1[%2]")
            .arg(m_aminoAcid.toString())
            .arg(QString::number(m_massDifference));
        }
    }
  else if(m_skipped)
    {
      return QString("[%1]").arg(m_aminoAcid.toString());
    }
  else
    {
      return m_aminoAcid.toString();
    }
}
PeptideModel::PeptideModel()
{
}

PeptideModel::~PeptideModel()
{
}

void
PeptideModel::reset()
{
  this->clear();
  m_cterShift     = 0;
  m_nterShift     = 0;
  m_precursorMass = 0;
}
void
PeptideModel::setCterShift(double mass_shift)
{
  m_cterShift = mass_shift;
}

void
PeptideModel::setNterShift(double mass_shift)
{
  m_nterShift = mass_shift;
}

void
PeptideModel::setPrecursorMass(double mass)
{
  m_precursorMass = mass;
}

double
PeptideModel::getPrecursorMass() const
{
  return m_precursorMass;
}
QString
PeptideModel::toProFormaMass(double mass)
{
  if(mass > 0)
    {
      return QString("+%1").arg(QString::number(mass, 'f', 4));
    }
  else
    {
      return QString("%1").arg(QString::number(mass, 'f', 4));
    }
}

double
PeptideModel::getMass() const
{
  double mass = m_cterShift + m_nterShift + MASSH2O;
  // qWarning() << mass;
  for(auto &aa : *this)
    {
      // qWarning() << aa.m_aminoAcid.toProForma();
      mass += aa.m_aminoAcid.getMass() + aa.m_massDifference;
    }
  return mass;
}

QString
PeptideModel::toProForma() const
{

  double mass_delta = m_precursorMass - getMass();

  QString proforma("[");
  if(mass_delta > 0)
    {
      proforma.append("+");
    }
  proforma.append(QString("%1]?").arg(QString::number(mass_delta, 'f', 4)));
  proforma.replace("[+0.0000]?", "");
  proforma.replace("[-0.0000]?", "");


  // QString proforma;

  if(size() > 0)
    {
      if(m_nterShift != 0)
        {
          proforma = '[' + toProFormaMass(m_nterShift) + "]-";
        }
      for(auto &aa : *this)
        {
          proforma.append(aa.toProForma());
        }
      if(m_cterShift != 0)
        {
          proforma = "-[" + toProFormaMass(m_cterShift) + ']';
        }
    }
  return proforma;
}
QString
PeptideModel::toInterpretation() const
{
  QString proforma;

  if(size() > 0)
    {
      if(m_nterShift > 0)
        {
          proforma = '[' + QString::number(m_nterShift) + "]";
        }
      for(auto &aa : *this)
        {
          proforma.append(aa.toInterpretation());
        }
      if(m_cterShift > 0)
        {
          proforma = "[" + QString::number(m_cterShift) + ']';
        }
    }
  return proforma;
}

} // namespace specpeptidoms
} // namespace pappso
